/*
 * Decompiled with CFR 0.152.
 */
package org.sing_group.seda.blast.transformation.dataset;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.sing_group.seda.blast.BlastUtils;
import org.sing_group.seda.blast.datatype.DatabaseQueryMode;
import org.sing_group.seda.blast.datatype.SequenceType;
import org.sing_group.seda.blast.datatype.blast.BlastType;
import org.sing_group.seda.blast.execution.BlastBinariesExecutor;
import org.sing_group.seda.core.execution.BinaryCheckException;
import org.sing_group.seda.datatype.DatatypeFactory;
import org.sing_group.seda.datatype.InDiskDatatypeFactory;
import org.sing_group.seda.datatype.InMemoryDatatypeFactory;
import org.sing_group.seda.datatype.Sequence;
import org.sing_group.seda.datatype.SequencesGroup;
import org.sing_group.seda.datatype.SequencesGroupDataset;
import org.sing_group.seda.io.FastaWriter;
import org.sing_group.seda.transformation.TransformationException;
import org.sing_group.seda.transformation.dataset.SequencesGroupDatasetTransformation;
import org.sing_group.seda.util.OsUtils;

public class BlastTransformation
implements SequencesGroupDatasetTransformation {
    public static final SequenceType DEFAULT_SEQUENCE_TYPE = SequenceType.NUCLEOTIDES;
    public static final BlastType DEFAULT_BLAST_TYPE = BlastType.BLASTN;
    public static final double DEFAULT_EVALUE = 0.05;
    public static final int DEFAULT_MAX_TARGET_SEQS = 500000;
    public static final boolean DEFAULT_EXTRACT_ONLY_HIT_REGIONS = false;
    public static final int DEFAULT_HIT_REGIONS_WINDOW_SIZE = 0;
    private BlastBinariesExecutor defaultBlastBinariesExecutor;
    private final SequenceType databaseType;
    private final BlastType blastType;
    private final DatabaseQueryMode databaseQueryMode;
    private final File databasesDirectory;
    private final Optional<File> aliasFile;
    private final double evalue;
    private final int maxTargetSeqs;
    private DatatypeFactory factory;
    private File queryFile;
    private String blastParams;
    private boolean extractOnlyHitRegions;
    private int hitRegionsWindowSize;
    private Path tempAliasDirectory;

    public BlastTransformation(BlastType blastType, DatabaseQueryMode databaseQueryMode, BlastBinariesExecutor blastBinariesExecutor, File queryFile, File databasesPath, double evalue, int maxTargetSeqs, boolean extractOnlyHitRegions, int hitRegionsWindowSize, String blastParams, DatatypeFactory factory) {
        this(blastType, databaseQueryMode, blastBinariesExecutor, queryFile, databasesPath, null, evalue, maxTargetSeqs, extractOnlyHitRegions, hitRegionsWindowSize, blastParams, factory);
    }

    public BlastTransformation(BlastType blastType, DatabaseQueryMode databaseQueryMode, BlastBinariesExecutor blastBinariesExecutor, File queryFile, File databasesPath, File aliasFile, double evalue, int maxTargetSeqs, boolean extractOnlyHitRegions, int hitRegionsWindowSize, String blastParams, DatatypeFactory factory) {
        this.databaseType = blastType.getDatabaseType();
        this.databaseQueryMode = databaseQueryMode;
        this.blastType = blastType;
        this.defaultBlastBinariesExecutor = blastBinariesExecutor;
        this.databasesDirectory = databasesPath;
        this.aliasFile = Optional.ofNullable(aliasFile);
        this.queryFile = queryFile;
        this.evalue = evalue;
        this.maxTargetSeqs = maxTargetSeqs;
        this.extractOnlyHitRegions = extractOnlyHitRegions;
        this.hitRegionsWindowSize = hitRegionsWindowSize;
        this.blastParams = blastParams;
        this.factory = factory;
        if (!this.isValidConfiguration()) {
            throw new RuntimeException("Invalid configuration");
        }
    }

    @Override
    public SequencesGroupDataset transform(SequencesGroupDataset dataset) throws TransformationException {
        Objects.requireNonNull(dataset);
        return this.blast(dataset);
    }

    private SequencesGroupDataset blast(SequencesGroupDataset dataset) {
        if (this.databaseQueryMode.equals((Object)DatabaseQueryMode.ALL)) {
            return this.factory.newSequencesGroupDataset(this.getSequenceGroups(this.blastDataset(dataset, "")));
        }
        LinkedList<File> sequenceResultFiles = new LinkedList<File>();
        for (SequencesGroup group : dataset.getSequencesGroups().collect(Collectors.toList())) {
            SequencesGroupDataset currentDataset = this.factory.newSequencesGroupDataset(group);
            sequenceResultFiles.addAll(this.blastDataset(currentDataset, "_" + group.getName()));
        }
        return this.factory.newSequencesGroupDataset(this.getSequenceGroups(sequenceResultFiles));
    }

    private List<File> blastDataset(SequencesGroupDataset dataset, String resultsSuffix) {
        List<File> blastResults;
        File aliasFile;
        List<File> blastDatabases = Collections.emptyList();
        try {
            blastDatabases = this.makeBlastDatabases(dataset, this.databasesDirectory);
        }
        catch (IOException | InterruptedException e) {
            throw new TransformationException("An error occurred while creating the databases");
        }
        try {
            aliasFile = this.aliasFile.orElse(this.getAliasTemporaryFile());
        }
        catch (IOException e) {
            throw new TransformationException("An error occurred while creating the alias");
        }
        try {
            this.makeBlastDatabasesAlias(blastDatabases, aliasFile);
        }
        catch (IOException | InterruptedException e) {
            throw new TransformationException("An error occurred while creating the alias");
        }
        try {
            blastResults = this.executeBlast(aliasFile);
        }
        catch (IOException | InterruptedException e1) {
            throw new TransformationException("An error occurred while running blast");
        }
        try {
            List<File> sequenceResultsFiles = this.extractSequences(blastResults, aliasFile, resultsSuffix);
            return sequenceResultsFiles;
        }
        catch (IOException | InterruptedException e) {
            throw new TransformationException("An error occurred while extracting result sequences");
        }
    }

    private File getAliasTemporaryFile() throws IOException {
        if (this.tempAliasDirectory == null) {
            this.tempAliasDirectory = Files.createTempDirectory("seda-blastdb-alias", new FileAttribute[0]);
            this.tempAliasDirectory.toFile().mkdir();
        }
        return Files.createTempFile(this.tempAliasDirectory, "alias", "", new FileAttribute[0]).toFile();
    }

    private SequencesGroup[] getSequenceGroups(List<File> sequenceResultsFiles) {
        LinkedList<SequencesGroup> groups = new LinkedList<SequencesGroup>();
        for (File result : sequenceResultsFiles) {
            groups.add(this.factory.newSequencesGroup(result.toPath()));
        }
        return groups.toArray(new SequencesGroup[groups.size()]);
    }

    private List<File> extractSequences(List<File> blastResults, File aliasFile, String fileSuffix) throws InterruptedException, IOException {
        LinkedList<File> sequenceFiles = new LinkedList<File>();
        for (File blastResult : blastResults) {
            if (this.extractOnlyHitRegions) {
                sequenceFiles.add(this.extractOnlyHitRegionsAsSequences(blastResult, aliasFile, fileSuffix));
                continue;
            }
            sequenceFiles.add(this.extractBatchSequences(blastResult, aliasFile, fileSuffix));
        }
        return sequenceFiles;
    }

    private File extractBatchSequences(File blastResult, File aliasFile, String fileSuffix) throws InterruptedException, IOException {
        File sequencesFile = new File(blastResult.getParentFile(), blastResult.getName().replace(".out", "") + fileSuffix + ".sequences");
        File blastSubjectList = this.extractSubjectList(blastResult);
        this.defaultBlastBinariesExecutor.blastDbCmd(aliasFile, blastSubjectList, sequencesFile);
        return sequencesFile;
    }

    private File extractSubjectList(File blastResult) throws IOException {
        File blastSubjectList = new File(blastResult.getParentFile(), blastResult.getName() + ".ids.txt");
        HashSet subjectIds = new HashSet();
        Files.readAllLines(blastResult.toPath()).forEach(l -> {
            String[] fields = l.split("\t");
            subjectIds.add(fields[1]);
        });
        Files.write(blastSubjectList.toPath(), subjectIds, new OpenOption[0]);
        return blastSubjectList;
    }

    private File extractOnlyHitRegionsAsSequences(File blastResult, File aliasFile, String fileSuffix) throws InterruptedException, IOException {
        List<BlastFormat6Hit> blastHits = BlastTransformation.extractBlastHits(blastResult);
        LinkedList<Sequence> sequences = new LinkedList<Sequence>();
        File sequencesFile = new File(blastResult.getParentFile(), blastResult.getName().replace(".out", "") + fileSuffix + ".sequences");
        InMemoryDatatypeFactory temporaryDatatypeFactory = new InMemoryDatatypeFactory();
        for (BlastFormat6Hit hit : blastHits) {
            int rangeStart = Math.min(hit.getSubjectStart(), hit.getSubjectEnd());
            int rangeEnd = Math.max(hit.getSubjectStart(), hit.getSubjectEnd()) + this.hitRegionsWindowSize;
            if (rangeStart - this.hitRegionsWindowSize > 0) {
                rangeStart -= this.hitRegionsWindowSize;
            }
            File temporaryHitFile = Files.createTempFile(blastResult.getParentFile().toPath(), "hit-" + hit.getSubjectSequenceId(), ".fasta", new FileAttribute[0]).toFile();
            this.defaultBlastBinariesExecutor.blastDbCmd(aliasFile, hit.getSubjectSequenceId(), new String(rangeStart + "-" + rangeEnd), temporaryHitFile);
            SequencesGroup subjectSequencesGroup = temporaryDatatypeFactory.newSequencesGroup(temporaryHitFile.toPath());
            if (subjectSequencesGroup.getSequenceCount() <= 0) continue;
            sequences.add(subjectSequencesGroup.getSequence(0));
        }
        FastaWriter.writeFasta(sequencesFile.toPath(), sequences.stream());
        return sequencesFile;
    }

    private static List<BlastFormat6Hit> extractBlastHits(File blastResult) throws IOException {
        LinkedList<BlastFormat6Hit> hits = new LinkedList<BlastFormat6Hit>();
        Files.readAllLines(blastResult.toPath()).forEach(l -> {
            String[] fields = l.split("\t");
            hits.add(new BlastFormat6Hit(fields[1], fields[8], fields[9]));
        });
        return hits;
    }

    private List<File> makeBlastDatabases(SequencesGroupDataset dataset, File databasesDirectory) throws IOException, InterruptedException {
        LinkedList<File> blastDatabases = new LinkedList<File>();
        for (SequencesGroup fasta : dataset.getSequencesGroups().collect(Collectors.toList())) {
            Path fastaFile = Files.createTempFile(fasta.getName() + "-", ".fasta", new FileAttribute[0]);
            FastaWriter.writeFasta(fastaFile, fasta.getSequences());
            File dbDirectory = new File(databasesDirectory, fasta.getName());
            dbDirectory.mkdir();
            File dbFile = new File(dbDirectory, fasta.getName());
            if (!BlastUtils.existDatabase(dbFile)) {
                this.makeblastdb(fastaFile.toFile(), dbFile);
            }
            blastDatabases.add(dbFile);
            fastaFile.toFile().delete();
        }
        return blastDatabases;
    }

    private void makeblastdb(File inFile, File dbFile) throws IOException, InterruptedException {
        this.defaultBlastBinariesExecutor.makeBlastDb(inFile, this.getBlastSequenceType(), dbFile, true);
    }

    private void makeBlastDatabasesAlias(List<File> blastDatabases, File outFile) throws IOException, InterruptedException {
        this.defaultBlastBinariesExecutor.makeDbAlias(blastDatabases, this.getBlastSequenceType(), outFile, "dbalias");
    }

    private List<File> executeBlast(File aliasFile) throws IOException, InterruptedException {
        SequencesGroup querySequences = new InDiskDatatypeFactory().newSequencesGroup(this.queryFile.toPath());
        LinkedList<File> blastResults = new LinkedList<File>();
        File blastDir = Files.createTempDirectory("seda-blast-results", new FileAttribute[0]).toFile();
        for (int i = 0; i < querySequences.getSequenceCount(); ++i) {
            Sequence querySequence = querySequences.getSequence(i);
            String querySequenceName = this.getQuerySequenceName(querySequence);
            File querySequenceFile = Files.createTempFile(blastDir.toPath(), querySequenceName, ".fasta", new FileAttribute[0]).toFile();
            FastaWriter.writeFasta(querySequenceFile.toPath(), querySequence);
            File output = this.executeBlast(this.blastType, blastDir, aliasFile, querySequenceFile, this.evalue, this.maxTargetSeqs, querySequenceName);
            blastResults.add(output);
        }
        return blastResults;
    }

    private String getQuerySequenceName(Sequence querySequence) {
        String name = querySequence.getName();
        for (String character : OsUtils.getInvalidOsFileCharacters()) {
            name.replace(character, "_");
        }
        return name;
    }

    private File executeBlast(BlastType blastType, File outDirectory, File database, File queryFile, double expectedValue, int maxTargetSeqs, String outputName) throws InterruptedException, IOException {
        File outFile = new File(outDirectory, outputName + ".out");
        if (!outDirectory.isDirectory() && !outDirectory.mkdirs()) {
            throw new IOException("Output directory could not be created: " + outDirectory);
        }
        this.defaultBlastBinariesExecutor.executeBlast(blastType, queryFile, database, expectedValue, maxTargetSeqs, outFile, "6", this.getAdditionalBlastParameters());
        return outFile;
    }

    private List<String> getAdditionalBlastParameters() {
        if (this.blastParams != null && !this.blastParams.isEmpty()) {
            return Arrays.asList(this.blastParams.split(" "));
        }
        return Collections.emptyList();
    }

    private String getBlastSequenceType() {
        return this.databaseType.getBlastName();
    }

    private boolean isValidConfiguration() {
        try {
            this.defaultBlastBinariesExecutor.checkBinary();
        }
        catch (BinaryCheckException e) {
            return false;
        }
        if (this.getQueryFile() == null) {
            return false;
        }
        if (this.getDatabasesDirectory() == null) {
            return false;
        }
        return !this.getAdditionalBlastParameters().contains("-evalue") && !this.getAdditionalBlastParameters().contains("max_target_seqs");
    }

    private File getDatabasesDirectory() {
        return this.databasesDirectory;
    }

    private File getQueryFile() {
        return this.queryFile;
    }

    private static class BlastFormat6Hit {
        private String subjectSequenceId;
        private String subjectStart;
        private String subjectEnd;

        BlastFormat6Hit(String subject, String subjectStart, String subjectEnd) {
            this.subjectSequenceId = subject;
            this.subjectStart = subjectStart;
            this.subjectEnd = subjectEnd;
        }

        public String getSubjectSequenceId() {
            return this.subjectSequenceId;
        }

        public int getSubjectEnd() {
            return Integer.valueOf(this.subjectEnd);
        }

        public int getSubjectStart() {
            return Integer.valueOf(this.subjectStart);
        }
    }
}

