package es.uvigo.esei.xcs.service;

import java.util.Date;
import static java.util.Objects.requireNonNull;

import java.security.Principal;
import java.util.List;

import javax.annotation.security.RolesAllowed;
import javax.ejb.EJB;
import javax.ejb.Stateless;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.MultidoseVaccine;
import es.uvigo.esei.xcs.domain.entities.PeriodicVaccine;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vaccination;
import es.uvigo.esei.xcs.domain.entities.Vaccine;

/**
 * EJB for managing Vaccinations. Access is restricted to VET role.
 * Provides CRUD operations and rules to check if a pet can be vaccinated.
 * 
 * @author Breixo Senra
 */
@Stateless
@RolesAllowed("VET")
public class VaccinationService {

    @Inject
    private Principal currentUser;

    @PersistenceContext
    private EntityManager em;

    @EJB
    private EmailService emailService;

    /**
     * Returns a vaccination by its ID.
     * 
     * @param vaccinationId the identifier of the vaccination.
     * @return the Vaccination entity, or {@code null} if not found.
     */
    public Vaccination get(int vaccinationId) {
        return em.find(Vaccination.class, vaccinationId);
    }

    /**
     * Returns a paginated list of vaccinations (0-based page index).
     * 
     * @param page the 0-based page index.
     * @param pageSize the maximum number of vaccinations per page.
     * @return a list of Vaccination entities.
     * @throws IllegalArgumentException if {@code page} is negative or {@code pageSize} is not positive.
     */
    public List<Vaccination> list(int page, int pageSize) {
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery("SELECT DISTINCT v FROM Vaccination v", Vaccination.class)
                 .setFirstResult(page * pageSize)
                 .setMaxResults(pageSize)
                 .getResultList();
    }

    /**
     * Creates a new vaccination for a pet with a given vaccine and date.
     * Sends an email notification to the pet's owner.
     * 
     * @param petId the identifier of the pet.
     * @param vaccineId the identifier of the vaccine.
     * @param date the date of the vaccination.
     * @return the persisted Vaccination entity.
     * @throws NullPointerException if {@code pet} or {@code vaccine} is null.
     * @throws IllegalArgumentException if the vaccination cannot be created due to vaccine rules.
     */
    public Vaccination create(Long petId, Long vaccineId, Date date) {
        Pet pet = requireNonNull(em.find(Pet.class, petId), "Pet can't be null");
        Vaccine vaccine = requireNonNull(em.find(Vaccine.class, vaccineId), "Vaccine can't be null");

        if (!canVaccinate(petId, vaccineId, date)) {
            throw new IllegalArgumentException("This vaccination cannot be created due to vaccine rules");
        }

        Vaccination vaccination = new Vaccination(pet, vaccine, date);
        em.persist(vaccination);
        emailService.send(
            pet.getOwner().getLogin(), 
            pet.getName() + " ha sido vacunado con " + vaccine.getName(), 
            pet.getName() + " ha sido vacunado con " + vaccine.getName()
        );
        return vaccination;
    }

    /**
     * Updates the date of an existing vaccination.
     * 
     * @param vaccinationId the identifier of the vaccination.
     * @param date the new date.
     * @return the updated Vaccination entity.
     * @throws NullPointerException if the vaccination does not exist.
     */
    public Vaccination updateDate(int vaccinationId, Date date) {
        Vaccination vaccination = em.find(Vaccination.class, vaccinationId);
        requireNonNull(vaccination, "Vaccination can't be null");
        vaccination.setDate(date);
        return em.merge(vaccination);
    }

    /**
     * Removes a vaccination by its ID.
     * 
     * @param vaccinationId the identifier of the vaccination.
     * @throws NullPointerException if the vaccination does not exist.
     */
    public void remove(int vaccinationId) {
        Vaccination vaccination = this.get(vaccinationId);
        requireNonNull(vaccination, "Vaccination can't be null");
        em.remove(vaccination);
    }

    /**
     * Checks whether a pet can be vaccinated with a given vaccine on a given date.
     * 
     * @param petId the identifier of the pet.
     * @param vaccineId the identifier of the vaccine.
     * @param date the intended vaccination date.
     * @return {@code true} if vaccination is allowed, {@code false} otherwise.
     */
    public Boolean canVaccinate(Long petId, Long vaccineId, Date date) {
        Pet pet = em.find(Pet.class, petId);
        Vaccine vaccine = em.find(Vaccine.class, vaccineId);
        if (pet == null || vaccine == null) return false;

        List<Vaccination> prevVaccinations = em.createQuery(
            "SELECT v FROM Vaccination v WHERE v.pet.id = :petId AND v.vaccine.id = :vaccineId ORDER BY v.date DESC",
            Vaccination.class)
            .setParameter("petId", petId)
            .setParameter("vaccineId", vaccineId)
            .getResultList();

        if (vaccine instanceof MultidoseVaccine) {
            MultidoseVaccine multi = (MultidoseVaccine) vaccine;
            Integer doses = multi.getDoses();
            if (doses == null) return false;
            return prevVaccinations.size() < doses;

        } else if (vaccine instanceof PeriodicVaccine) {
            PeriodicVaccine periodic = (PeriodicVaccine) vaccine;
            if (prevVaccinations.isEmpty()) return true;
            Vaccination last = prevVaccinations.get(0);
            if (last.getDate() == null || date == null) return false;

            long diffDays;
            switch (periodic.getPeriodicType()) {
                case YEARS:  diffDays = periodic.getPeriode() * 365L; break;
                case MONTHS: diffDays = periodic.getPeriode() * 30L; break;
                case DAYS:   diffDays = periodic.getPeriode(); break;
                default:     return false;
            }

            long diffMillis = date.getTime() - last.getDate().getTime();
            return diffMillis >= diffDays * 24L * 60L * 60L * 1000L;

        } else { // Monodose
            return prevVaccinations.isEmpty();
        }
    }
}
