package es.uvigo.esei.xcs.rest;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import javax.ejb.EJB;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import es.uvigo.esei.xcs.domain.entities.Vaccination;
import es.uvigo.esei.xcs.service.VaccinationService;

/**
 * REST resource for managing pet vaccinations.
 * Provides endpoints to check if a vaccination can be applied and to create vaccinations.
 * Accessible under the path "/vaccination".
 * 
 * @author Breixo
 */
@Path("vaccination")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class VaccinationResource {

    @EJB
    private VaccinationService vaccinationService;

    /**
     * Checks if a pet can be vaccinated with a specific vaccine today.
     * 
     * @param petId the ID of the pet.
     * @param vaccineId the ID of the vaccine.
     * @return HTTP 200 with a boolean indicating if vaccination is allowed.
     */
    @GET
    @Path("canVaccinate/{petId}/{vaccineId}")
    public Response canVaccinate(@PathParam("petId") Long petId,
                                 @PathParam("vaccineId") Long vaccineId) {
        Date now = new Date();
        Boolean result = vaccinationService.canVaccinate(petId, vaccineId, now);
        return Response.ok(result).build();
    }

    /**
     * Creates a vaccination for a pet with a given vaccine.
     * 
     * @param petId the ID of the pet.
     * @param vaccineId the ID of the vaccine.
     * @param textDate optional date in yyyy-MM-dd format. Defaults to today if null.
     * @return HTTP 200 with the created Vaccination, or HTTP 400 if invalid input.
     */
    @POST
    @Path("create/{petId}/{vaccineId}")
    public Response createVaccination(@PathParam("petId") Long petId,
                                      @PathParam("vaccineId") Long vaccineId,
                                      @QueryParam("date") String textDate) {
        Date date = new Date();
        if (textDate != null) {
            try {
                date = new SimpleDateFormat("yyyy-MM-dd").parse(textDate);
            } catch (ParseException e) {
                return Response.status(Response.Status.BAD_REQUEST)
                               .entity("Invalid date format, expected yyyy-MM-dd")
                               .build();
            }
        }

        try {
            Vaccination vaccination = vaccinationService.create(petId, vaccineId, date);
            return Response.ok(vaccination).build();
        } catch (IllegalArgumentException e) {
            return Response.status(Response.Status.BAD_REQUEST)
                           .entity(e.getMessage())
                           .build();
        }
    }
}
