package es.uvigo.esei.xcs.rest;

import java.net.URI;
import javax.ejb.EJB;
import javax.ejb.EJBAccessException;
import javax.persistence.EntityExistsException;
import javax.ws.rs.*;
import javax.ws.rs.core.*;

import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.rest.entity.PetData;
import es.uvigo.esei.xcs.service.PetService;

/**
 * REST resource that manages pets in the application.
 * Provides CRUD operations, pagination, and access control.
 * Accessible under the path "/pet".
 * 
 * Only the pet's owner or authorized roles can modify or delete a pet.
 * Access violations throw SecurityException.
 * 
 * @author Breixo
 */
@Path("pet")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class PetResource {

	@EJB
	private PetService service;

	@Context
	private UriInfo uriInfo;

	/**
	 * Returns a pet by its identifier.
	 * 
	 * @param id pet ID.
	 * @return HTTP 200 with pet data.
	 * @throws SecurityException if current user is not the pet's owner.
	 * @throws IllegalArgumentException if pet not found.
	 */
	@Path("{id}")
	@GET
	public Response get(@PathParam("id") Long id) throws SecurityException {
		try {
			final Pet pet = this.service.get(id);
			if (pet == null)
				throw new IllegalArgumentException("Pet not found: " + id);
			return Response.ok(pet).build();
		} catch (EJBAccessException eae) {
			throw new SecurityException(eae);
		}
	}

	/**
	 * Returns a paginated list of all pets.
	 * 
	 * @param page 0-based page index.
	 * @param pageSize number of results per page.
	 * @return HTTP 200 with list of pets.
	 */
	@GET
	@Path("lista")
	public Response getAll(@QueryParam("page") @DefaultValue("0") int page,
						   @QueryParam("pageSize") @DefaultValue("10") int pageSize) {
		return Response.ok(this.service.getAll(page, pageSize)).build();
	}

	/**
	 * Returns a paginated list of pets owned by the current user.
	 * 
	 * @param page 0-based page index.
	 * @param pageSize number of results per page.
	 * @return HTTP 200 with list of current owner's pets.
	 */
	@GET
	public Response list(@QueryParam("page") @DefaultValue("0") int page,
						 @QueryParam("pageSize") @DefaultValue("10") int pageSize) {
		return Response.ok(this.service.list(page, pageSize)).build();
	}

	/**
	 * Creates a new pet for the current user.
	 * 
	 * @param petData data for the new pet.
	 * @return HTTP 201 with Location header pointing to the new pet.
	 * @throws IllegalArgumentException if petData is null or pet already exists.
	 * @throws SecurityException if current user cannot own this pet.
	 */
	@POST
	public Response create(PetData petData) throws SecurityException {
		if (petData == null)
			throw new IllegalArgumentException("pet can't be null");

		try {
			final Pet pet = this.service.create(petData.toPet());
			final URI petUri = uriInfo.getAbsolutePathBuilder()
				.path(String.valueOf(pet.getId()))
				.build();
			return Response.created(petUri).build();
		} catch (EntityExistsException eee) {
			throw new IllegalArgumentException("The pet already exists");
		} catch (EJBAccessException eae) {
			throw new SecurityException(eae);
		}
	}

	/**
	 * Updates an existing pet's data.
	 * 
	 * @param id pet ID.
	 * @param petData updated pet data.
	 * @return HTTP 200 when updated successfully.
	 * @throws IllegalArgumentException if petData is null or pet has no owner.
	 * @throws SecurityException if current user is not pet's owner.
	 */
	@Path("{id}")
	@PUT
	public Response update(@PathParam("id") Long id, PetData petData) throws SecurityException {
		if (petData == null)
			throw new IllegalArgumentException("pet can't be null");

		try {
			final Pet pet = this.service.get(id);
			petData.assignData(pet);
			this.service.update(pet);
			return Response.ok().build();
		} catch (EJBAccessException eae) {
			throw new SecurityException(eae);
		}
	}

	/**
	 * Deletes a pet by its ID.
	 * 
	 * @param id pet ID.
	 * @return HTTP 200 when deleted successfully.
	 * @throws IllegalArgumentException if pet does not exist.
	 * @throws SecurityException if current user is not pet's owner.
	 */
	@Path("{id}")
	@DELETE
	public Response delete(@PathParam("id") Long id) throws SecurityException {
		try {
			this.service.remove(id);
			return Response.ok().build();
		} catch (EJBAccessException eae) {
			throw new SecurityException(eae);
		}
	}
}
