package es.uvigo.esei.xcs.rest;

import java.net.URI;
import javax.ejb.EJB;
import javax.persistence.EntityExistsException;
import javax.ws.rs.*;
import javax.ws.rs.core.*;

import es.uvigo.esei.xcs.domain.entities.IdentifierType;
import es.uvigo.esei.xcs.domain.entities.Owner;
import es.uvigo.esei.xcs.rest.entity.OwnerEditionData;
import es.uvigo.esei.xcs.rest.entity.OwnerCreationData;
import es.uvigo.esei.xcs.service.OwnerService;
import static java.util.Objects.requireNonNull;

/**
 * REST resource that manages owners and their related pets and vaccinations.
 * Provides CRUD operations and pagination.
 * 
 * Accessible under the path "/owner".
 * 
 * @author Breixo
 */
@Path("owner")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class OwnerResource {

	@EJB
	private OwnerService service;

	@Context
	private UriInfo uriInfo;

	/**
	 * Returns the owner identified by the given login.
	 * 
	 * @param login unique login of the owner.
	 * @return HTTP 200 with the owner data.
	 * @throws IllegalArgumentException if the login is null or not found.
	 */
	@GET
	@Path("{login}")
	public Response get(@PathParam("login") String login) {
		if (login == null)
			throw new IllegalArgumentException("login can't be null");

		final Owner owner = this.service.get(login);
		if (owner == null)
			throw new IllegalArgumentException("Owner not found: " + login);

		return Response.ok(owner).build();
	}

	/**
	 * Returns a paginated list of owners.
	 * 
	 * @param page 0-based page index.
	 * @param pageSize number of results per page.
	 * @return HTTP 200 with the list of owners.
	 */
	@GET
	public Response list(@QueryParam("page") int page, @QueryParam("pageSize") int pageSize) {
		return Response.ok(this.service.list(page, pageSize)).build();
	}

	/**
	 * Creates a new owner (and optionally its pets).
	 * 
	 * @param ownerData data of the new owner.
	 * @return HTTP 201 with the URI of the created owner.
	 * @throws IllegalArgumentException if the data is null or already exists.
	 */
	@POST
	public Response create(OwnerCreationData ownerData) {
		if (ownerData == null)
			throw new IllegalArgumentException("ownerData can't be null");

		try {
			final Owner newOwner = this.service.create(ownerData.toOwner());
			final URI ownerUri = uriInfo.getAbsolutePathBuilder()
				.path(newOwner.getLogin())
				.build();
			return Response.created(ownerUri).build();
		} catch (EntityExistsException eee) {
			throw new IllegalArgumentException("The owner already exists");
		}
	}

	/**
	 * Updates an existing owner and its pets.
	 * 
	 * @param login owner login.
	 * @param ownerData updated owner data.
	 * @return HTTP 200 when updated successfully.
	 * @throws IllegalArgumentException if login or data are null.
	 */
	@PUT
	@Path("{login}")
	public Response update(@PathParam("login") String login, OwnerEditionData ownerData) {
		requireNonNull(login, "login can't be null");
		requireNonNull(ownerData, "ownerData can't be null");

		final Owner owner = this.service.get(login);
		ownerData.assignData(owner);
		this.service.update(owner);
		return Response.ok().build();
	}

	/**
	 * Deletes the owner identified by the given login.
	 * 
	 * @param login owner login.
	 * @return HTTP 200 when deletion succeeds.
	 * @throws IllegalArgumentException if login is null.
	 */
	@DELETE
	@Path("{login}")
	public Response delete(@PathParam("login") String login) {
		if (login == null)
			throw new IllegalArgumentException("login can't be null");
		this.service.remove(login);
		return Response.ok().build();
	}

	/**
	 * Returns the vaccinations for a pet identified by type and value.
	 * 
	 * @param petIdentifierType type of identifier.
	 * @param petIdentifierValue value of the identifier.
	 * @param page 0-based page index.
	 * @param pageSize number of results per page.
	 * @return HTTP 200 with the list of vaccinations.
	 */
	@GET
	@Path("pet/{petIdentifierType}/{petIdentifierValue}/vaccination")
	public Response listVaccinations(
		@PathParam("petIdentifierType") IdentifierType petIdentifierType,
		@PathParam("petIdentifierValue") String petIdentifierValue,
		@QueryParam("page") int page,
		@QueryParam("pageSize") int pageSize) {

		return Response.ok(this.service.getVaccinationsFromOwnPet(
			petIdentifierType,
			petIdentifierValue,
			page,
			pageSize
		)).build();
	}
}
