package es.uvigo.esei.xcs.service;

import static java.util.Objects.requireNonNull;

import java.io.Console;
import java.security.Principal;
import java.util.List;

import javax.annotation.security.PermitAll;
import javax.annotation.security.RolesAllowed;
import javax.ejb.EJB;
import javax.ejb.EJBAccessException;
import javax.ejb.Stateless;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.Owner;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.domain.entities.Vaccination;
import es.uvigo.esei.xcs.domain.entities.Vaccine;
import es.uvigo.esei.xcs.domain.entities.Vet;

/**
 * EJB for the Pets. Only owners have access to this class, and only to their
 * own pets.
 * 
 * @author Miguel Reboiro Jato
 */
@Stateless
//@RolesAllowed("VET")
@PermitAll
public class PetService {
	@Inject
	private Principal currentUser;
	
	@EJB
	private EmailService emailService;
	
	@PersistenceContext
	private EntityManager em;
	
	/**
	 * Returns a pet identified by the provided id. If an owner tries to access
	 * a pet that does now own, an {@link EJBAccessException} will be thrown.
	 * 
	 * @param id the identified of a pet.
	 * @return a pet identified by the provided identifier or {@code null} if no
	 * pet exists with the provided identifier.
	 * @throws EJBAccessException if the current owner does not owns the pet.
	 */
	/*public Pet get(Long id) {
		final Pet pet = em.find(Pet.class, id);

		if (pet == null) {
			return null;
		} else if (pet.getOwner().getLogin().equals(this.currentOwner.getName())) {
			return pet;
		} else {
			throw new EJBAccessException("Pet's owner is not the current principal");
		}
	}*/
	public int countAll() {
	    Long count = em.createQuery("SELECT COUNT(p) FROM Pet p", Long.class)
	                   .getSingleResult();
	    return count.intValue();
	}
	
	
	public Pet get(Long id) {
		return em.find(Pet.class, id);
	}
	
	/*public List<Pet> getAll(int page, int pageSize) {
		if (page < 0) {
			throw new IllegalArgumentException("The page can't be negative");
		}
		if (pageSize <= 0) {
			throw new IllegalArgumentException("The page size can't be negative or zero");
		}
	    return em.createQuery("SELECT p FROM Pet p", Pet.class)
	    		.setFirstResult(page * pageSize)
				.setMaxResults(pageSize)
				.getResultList();
	}*/
	
	public List<Pet> getAll(int first, int pageSize) {
	    if (first < 0) throw new IllegalArgumentException("First can't be negative");
	    if (pageSize <= 0) throw new IllegalArgumentException("Page size must be positive");

	    return em.createQuery("SELECT p FROM Pet p", Pet.class)
	             .setFirstResult(first)   // no multiplicar por pageSize
	             .setMaxResults(pageSize)
	             .getResultList();
	}


	
	/**
	 * Returns the complete list of pets of the current owner.
	 *  
	 * @return the complete list of pets of the current owner. 
	 */
	public List<Pet> list(int page, int pageSize) {
		if (page < 0) {
			throw new IllegalArgumentException("The page can't be negative");
		}
		if (pageSize <= 0) {
			throw new IllegalArgumentException("The page size can't be negative or zero");
		}
		return em.createQuery("SELECT p FROM Pet p WHERE p.owner.login = :login", Pet.class)
			.setFirstResult(page * pageSize)
			.setMaxResults(pageSize)
			.setParameter("login", currentUser.getName())
			.getResultList();
	}
	
	/**
	 * Creates a new pet owned by the current user.
	 * 
	 * @param pet a new pet to be stored.
	 * @return the persistent version of the pet created.
	 * @throws EJBAccessException if the pet already has an owner and it is not
	 * the current user. If the pet has no owner, this exception will be never
	 * thrown.
	 * @throws IllegalArgumentException if a pet with the same identifier
	 * already exists.
	 */
	public Pet create(Pet pet) {
		requireNonNull(pet, "Pet can't be null");
		
		final Owner owner = em.find(Owner.class, currentUser.getName());
		
		if (pet.getOwner() != null && !pet.getOwner().getLogin().equals(owner.getLogin())) {
			throw new EJBAccessException("Pet's owner is not the current principal");
		} else {
			pet.setOwner(owner);
			
			this.em.persist(pet);
			
			return pet;
		}
	}
	
	/**
	 * Updates the information of a pet. If the pet is not stored, it will be
	 * created.
	 * 
	 * @param pet a pet to be updated.
	 * @return the updated pet.
	 * @throws IllegalArgumentException if the pet has no owner.
	 * @throws EJBAccessException if the pet's owner is not the current user.
	 */
	public Pet update(Pet pet) {
		if (pet.getOwner() == null)
			throw new IllegalArgumentException("Pet must have an owner");
		
		if (pet.getOwner().getLogin().equals(this.currentUser.getName())) {
			return em.merge(pet);
		} else {
			throw new EJBAccessException("Pet's owner is not the current principal");
		}
	}
	
	/**
	 * Deletes a pet.
	 * 
	 * @param id the identifier of the pet to be deleted.
	 * @throws IllegalArgumentException if there is no pet with the provided
	 * identifier.
	 * @throws EJBAccessException if the pet's owner is not the current user.
	 */
	public void remove(Long id) {
		final Pet pet = this.get(id);
		pet.setOwner(null);
		
		em.remove(pet);
	}
	
	public List<Vaccine> getVaccinesByPetId(Long id, int page, int pageSize){
		if (page < 0) {
			throw new IllegalArgumentException("The page can't be negative");
		}
		if (pageSize <= 0) {
			throw new IllegalArgumentException("The page size can't be negative or zero");
		}
		return em.createQuery("SELECT v.pet FROM Vaccination v WHERE v.pet.id = :id", Vaccine.class)
				.setFirstResult(page * pageSize)
				.setMaxResults(pageSize)
				.getResultList();
	}
	
	
	public void assignVetToPet(Long petId) {
	    requireNonNull(petId, "Pet ID can't be null");
	    //requireNonNull(vetLogin, "Vet login can't be null");

	    Pet pet = em.find(Pet.class, petId);
	    if (pet == null)
	        throw new IllegalArgumentException("Pet not found");

	    Vet vet = em.find(Vet.class, currentUser.getName());
	    if (vet == null)
	        throw new IllegalArgumentException("Vet not found");

	    pet.addVet(vet);
	    pet.internalAddVet(vet);
	    em.merge(pet);
	}

	
	public void unassignVetFromPet(Long petId) {
	    requireNonNull(petId, "Pet ID can't be null");
	    //requireNonNull(vetLogin, "Vet login can't be null");

	    Pet pet = em.find(Pet.class, petId);
	    if (pet == null)
	        throw new IllegalArgumentException("Pet not found");

	    Vet vet = em.find(Vet.class, currentUser.getName());
	    if (vet == null)
	        throw new IllegalArgumentException("Vet not found");

	    pet.removeVet(vet);
	    pet.internalRemoveVet(vet);
	    em.merge(pet);
	}

	public Principal getCurrentUser() {
	    return this.currentUser;
	}
	
	public boolean isAssignedToCurrentVet(Long petId) {
	    requireNonNull(petId, "Pet ID can't be null");

	    
	    Long count = em.createQuery(
	        "SELECT COUNT(p) FROM Pet p JOIN p.vets v WHERE p.id = :petId AND v.login = :login",
	        Long.class
	    )
	    .setParameter("petId", petId)
	    .setParameter("login", currentUser.getName())
	    .getSingleResult();

	    return count > 0;
	}
	
}
