package es.uvigo.esei.xcs.jsf;

import static java.util.stream.Collectors.joining;

import java.util.List;
import java.util.Map;

import javax.annotation.PostConstruct;
import javax.enterprise.context.RequestScoped;
import javax.faces.context.FacesContext;
import javax.inject.Inject;
import javax.inject.Named;

import es.uvigo.esei.xcs.domain.entities.Owner;
import es.uvigo.esei.xcs.domain.entities.Pet;
import es.uvigo.esei.xcs.service.OwnerService;

import org.primefaces.model.LazyDataModel;
import org.primefaces.model.SortMeta;
import org.primefaces.model.FilterMeta;

@Named("owner")
@RequestScoped
public class OwnerManagedBean {
    @Inject
    private OwnerService service;

    private String login;
    private String password;
    private boolean editing;
    private String errorMessage;

    private LazyDataModel<Owner> owners;

    @PostConstruct
    public void init() {
        owners = new LazyDataModel<Owner>() {
            @Override
            public List<Owner> load(int first, int pageSize, Map<String, SortMeta> sortBy, Map<String, FilterMeta> filterBy) {
                return service.list(first, pageSize);
            }

            @Override
            public int count(Map<String, FilterMeta> filterBy) {
                return service.countAll();
            }
        };
    }

    public LazyDataModel<Owner> getOwners() {
        return owners;
    }

    public String getLogin() { return login; }
    public void setLogin(String login) { this.login = login; }
    public String getPassword() { return password; }
    public void setPassword(String password) { this.password = password; }
    public boolean isEditing() { return editing; }
    public void setEditing(boolean editing) { this.editing = editing; }
    public String getErrorMessage() { return errorMessage; }
    public boolean isError() { return errorMessage != null; }

    public String getPetNames(String login) {
        return this.service.getPets(login, 0, 100).stream()
                .map(Pet::getName)
                .collect(joining(", "));
    }

    public String edit(String login) {
        this.editing = true;
        this.login = login;
        return getViewId();
    }

    public String cancelEditing() {
        clear();
        return getViewId();
    }

    public String remove(String login) {
        service.remove(login);
        return redirectTo(getViewId());
    }

    public String store() {
        try {
            if (isEditing()) {
                Owner owner = service.get(this.login);
                owner.changePassword(this.password);
                service.update(owner);
            } else {
                service.create(new Owner(login, password));
            }
            clear();
            return redirectTo(getViewId());
        } catch (Throwable t) {
            this.errorMessage = t.getMessage();
            return getViewId();
        }
    }

    private void clear() {
        this.login = null;
        this.password = null;
        this.errorMessage = null;
        this.editing = false;
    }

    private String redirectTo(String url) { return url + "?faces-redirect=true"; }
    private String getViewId() { return FacesContext.getCurrentInstance().getViewRoot().getViewId(); }
}
