package es.uvigo.esei.xcs.service;

import static java.util.Objects.requireNonNull;
import java.util.List;

import javax.annotation.security.RolesAllowed;
import javax.ejb.Stateless;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import es.uvigo.esei.xcs.domain.entities.Administrator;
import es.uvigo.esei.xcs.domain.entities.User;

/**
 * EJB for managing Administrator entities.
 * Provides CRUD operations and paginated user listing.
 * 
 * Accessible only to ADMIN users.
 * 
 * @author Breixo Senra
 */
@Stateless
@RolesAllowed("ADMIN")
public class AdministratorService {

    @PersistenceContext
    private EntityManager em;

    /**
     * Returns an administrator by its ID.
     */
    public Administrator get(int id) {
        return em.find(Administrator.class, id);
    }

    /**
     * Returns a paginated list of all users.
     */
    public List<User> list(int page, int pageSize) {
        if (page < 0) throw new IllegalArgumentException("The page can't be negative");
        if (pageSize <= 0) throw new IllegalArgumentException("The page size can't be negative or zero");

        return em.createQuery("SELECT DISTINCT u FROM User u", User.class)
                 .setFirstResult(page * pageSize)
                 .setMaxResults(pageSize)
                 .getResultList();
    }

    /**
     * Persists a new administrator entity.
     */
    public Administrator create(Administrator administrator) {
        requireNonNull(administrator, "Administrator can't be null");
        em.persist(administrator);
        return administrator;
    }

    /**
     * Updates an existing administrator.
     */
    public Administrator update(Administrator administrator) {
        requireNonNull(administrator, "Administrator can't be null");
        return em.merge(administrator);
    }

    /**
     * Removes an administrator by ID.
     */
    public void remove(int id) {
        em.remove(this.get(id));
    }
}
