package es.uvigo.esei.xcs.rest;

import static java.util.Objects.requireNonNull;
import java.net.URI;
import javax.ejb.EJB;
import javax.ws.rs.*;
import javax.ws.rs.core.*;

import es.uvigo.esei.xcs.domain.entities.Vaccine;
import es.uvigo.esei.xcs.rest.entity.VaccineCreationData;
import es.uvigo.esei.xcs.rest.entity.VaccineEditionData;
import es.uvigo.esei.xcs.service.VaccineService;

/**
 * REST resource for managing vaccines.
 * Provides endpoints to list, create, update, and delete vaccines.
 */
@Path("vaccine")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class VaccineResource {

    @EJB
    private VaccineService vaccineService;

    @Context
    private UriInfo uriInfo;

    /**
     * Returns a paginated list of vaccines.
     * 
     * @param page 0-based page index.
     * @param pageSize number of items per page.
     * @return HTTP 200 with the list of vaccines.
     */
    @GET
    public Response list(@QueryParam("page") @DefaultValue("0") int page,
                         @QueryParam("pageSize") @DefaultValue("10") int pageSize) {
        return Response.ok(vaccineService.list(page, pageSize)).build();
    }

    /**
     * Creates a new vaccine.
     * 
     * @param vaccineData data to create the vaccine.
     * @return HTTP 201 with the URI of the created vaccine.
     * @throws IllegalArgumentException if vaccineData is null.
     */
    @POST
    public Response create(VaccineCreationData vaccineData) {
        requireNonNull(vaccineData, "vaccineData can't be null");
        Vaccine vaccine = vaccineService.create(
        	vaccineData.getName(),
        	vaccineData.getType(),
        	vaccineData.getDoses(),
        	vaccineData.getPeriodicType(),
        	vaccineData.getPeriode()
        );
        URI uri = uriInfo.getAbsolutePathBuilder().path(String.valueOf(vaccine.getId())).build();
        return Response.created(uri).build();
    }

    /**
     * Updates only the name of an existing vaccine.
     * 
     * @param id vaccine ID.
     * @param newName new name for the vaccine.
     * @return HTTP 200 with the updated vaccine.
     * @throws IllegalArgumentException if newName is null.
     */
    @PUT
    @Path("{id}/name")
    public Response updateName(@PathParam("id") Long id, String newName) {
        requireNonNull(newName, "newName can't be null");
        Vaccine updated = vaccineService.updateName(id, newName);
        return Response.ok(updated).build();
    }

    /**
     * Updates all editable fields of an existing vaccine.
     * 
     * @param id vaccine ID.
     * @param vaccineData data to update.
     * @return HTTP 200 when updated successfully.
     * @throws IllegalArgumentException if vaccineData is null.
     */
    @PUT
    @Path("{id}")
    public Response update(@PathParam("id") Long id, VaccineEditionData vaccineData) {
        requireNonNull(vaccineData, "vaccineData can't be null");
        Vaccine vaccine = vaccineService.get(id);
        vaccineData.assignData(vaccine);
        vaccineService.update(vaccine);
        return Response.ok().build();
    }

    /**
     * Deletes a vaccine by its ID.
     * 
     * @param id vaccine ID.
     * @return HTTP 200 when deleted successfully.
     * @throws IllegalArgumentException if vaccine does not exist.
     */
    @DELETE
    @Path("{id}")
    public Response delete(@PathParam("id") Long id) {
        vaccineService.remove(id);
        return Response.ok().build();
    }
}
