package es.uvigo.esei.xcs.domain.entities;

import static java.util.Arrays.stream;
import static java.util.Collections.unmodifiableCollection;
import static java.util.Objects.requireNonNull;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.OneToMany;
import javax.xml.bind.annotation.XmlRootElement;

@Entity
@DiscriminatorValue("OWNER")
@XmlRootElement(name = "owner", namespace = "http://entities.domain.xcs.esei.uvigo.es")
public class Owner extends User implements Serializable {
	private static final long serialVersionUID = 1L;
	
	@OneToMany(
		mappedBy = "owner",
		cascade = CascadeType.ALL,
		orphanRemoval = true,
		fetch = FetchType.EAGER
	)
	private Set<Pet> pets;
	
	public Owner() {}
	
	public Owner(String login, String password) {
		super(login, password);
		this.role = "OWNER";
		this.pets = new HashSet<>();
	}
	
	Owner(String login, String password, Pet ... pets) {
		this(login, password);
		
		stream(pets).forEach(this::addPet);
	}
	
	public Collection<Pet> getPets() {
		return unmodifiableCollection(pets);
	}
	
	public void addPet(Pet pet) {
		requireNonNull(pet, "pet can't be null");
		
		if (!this.ownsPet(pet)) {
			pet.setOwner(this);
		}
	}
	
	public void removePet(Pet pet) {
		requireNonNull(pet, "pet can't be null");
		
		if (this.ownsPet(pet)) {
			pet.setOwner(null);
		} else {
			throw new IllegalArgumentException("pet doesn't belong to this owner");
		}
	}
	
	public boolean ownsPet(Pet pet) {
		return this.pets.contains(pet);
	}
	
	void internalAddPet(Pet pet) {
		requireNonNull(pet, "pet can't be null");
		
		if (!this.ownsPet(pet))
			this.pets.add(pet);
	}
	
	void internalRemovePet(Pet pet) {
		requireNonNull(pet, "pet can't be null");
		
		this.pets.remove(pet);
	}
	
}
